import os
import time
from azure.ai.agents import AgentsClient
from azure.ai.agents.models import ListSortOrder
from azure.identity import DefaultAzureCredential
from dotenv import load_dotenv

# Load environment variables from .env file
load_dotenv()

# Get configuration from environment variables
endpoint = os.getenv("AZURE_AI_ENDPOINT")
model_deployment = os.getenv("MODEL_DEPLOYMENT_NAME")

# Create the AgentsClient with authentication
agents_client = AgentsClient(
    endpoint=endpoint,
    credential=DefaultAzureCredential(exclude_environment_credential=True)
)

print("Connected to Azure AI Foundry")

# Create an agent
agent = agents_client.create_agent(
    model=model_deployment,
    name="my-agent",
    instructions="You are a helpful agent"
)

print(f"Created agent: {agent.id}")

# Create a thread (conversation)
thread = agents_client.threads.create()
print(f"Created thread: {thread.id}")

# Send a message to the agent
message = agents_client.messages.create(
    thread_id=thread.id,
    role="user",
    content="What is the stock price of Microsoft?"
)
print(f"Created message: {message.id}")

# Run the agent to process the message
run = agents_client.runs.create(
    thread_id=thread.id,
    agent_id=agent.id
)

print(f"Started run: {run.id}")

# Poll until the run completes
while run.status in ["queued", "in_progress", "requires_action"]:
    time.sleep(1)
    run = agents_client.runs.get(thread_id=thread.id, run_id=run.id)
    print(f"Run status: {run.status}")

print(f"Run completed with status: {run.status}")

# Get all messages from the thread
messages = agents_client.messages.list(
    thread_id=thread.id,
    order=ListSortOrder.ASCENDING
)

# Print the conversation
print("\n=== Conversation ===")
for msg in messages:
    if msg.text_messages:
        last_text = msg.text_messages[-1]
        print(f"{msg.role}: {last_text.text.value}")